
//*****************************************************************************
//
// Author: Stephen Bolanos
// Contact: bstephen94@gmail.com
// Date written: 2015 June 14
//
// The code below configures Timer 2 and Timer 3 for use of main program.
// 
//*****************************************************************************

#include <stdint.h>
#include <stdbool.h>

#include "inc/hw_ints.h"
#include "inc/hw_memmap.h"
#include "inc/hw_types.h"

#include "driverlib/gpio.h"
#include "driverlib/interrupt.h"
#include "driverlib/pin_map.h"
#include "driverlib/rom.h"
#include "driverlib/rom_map.h"
#include "driverlib/sysctl.h"
#include "driverlib/pwm.h"
#include "driverlib/timer.h"

uint32_t g_ui32SysClock;            // System clock rate in Hz

//*****************************************************************************
//
// Initialize the two timers used for measuring the duty cycle
//
//*****************************************************************************
void
two_timer_init(void)
{
    // Set the clocking to run directly from the crystal at 120MHz.
    g_ui32SysClock = MAP_SysCtlClockFreqSet((SYSCTL_XTAL_25MHZ | SYSCTL_OSC_MAIN | SYSCTL_USE_PLL | SYSCTL_CFG_VCO_480), 120000000);

    // Enable the GPIO port that is used for the input edge counting
    MAP_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOM);

    // Configure PM2 as Timer 3A
    MAP_GPIOPinTypeTimer(GPIO_PORTM_BASE, GPIO_PIN_2);
    MAP_GPIOPinConfigure(GPIO_PM2_T3CCP0);
    
    // Configure PM1 as the negative edge capture on Timer 2B
    MAP_GPIOPinTypeTimer(GPIO_PORTM_BASE, GPIO_PIN_1);
    MAP_GPIOPinConfigure(GPIO_PM1_T2CCP1);

    // Configure PM0 as the positive edge capture on Timer 2A
    MAP_GPIOPinTypeTimer(GPIO_PORTM_BASE, GPIO_PIN_0);
    MAP_GPIOPinConfigure(GPIO_PM0_T2CCP0);

    // Enable the timer peripheral used by this example
    MAP_SysCtlPeripheralEnable(SYSCTL_PERIPH_TIMER0);       // For use with the TimerSynchronize()
    MAP_SysCtlPeripheralEnable(SYSCTL_PERIPH_TIMER2);       // For use with edge-time mode
    MAP_SysCtlPeripheralEnable(SYSCTL_PERIPH_TIMER3);       // For use with timeout events for rollover monitoring
    
    // Enable processor interrupts.
    MAP_IntMasterEnable();

    // Configure Timer 3 as a 16-bit periodic timer to monitor rollovers
    MAP_TimerConfigure(TIMER3_BASE, (TIMER_CFG_SPLIT_PAIR | TIMER_CFG_A_PERIODIC));

    // Configure Timer 4A and B to trigger edge-time mode on edge events
    MAP_TimerConfigure(TIMER2_BASE, (TIMER_CFG_SPLIT_PAIR | TIMER_CFG_A_CAP_TIME | TIMER_CFG_B_CAP_TIME));
    MAP_TimerControlEvent(TIMER2_BASE, TIMER_A, TIMER_EVENT_POS_EDGE);
    MAP_TimerControlEvent(TIMER2_BASE, TIMER_B, TIMER_EVENT_NEG_EDGE);

    MAP_TimerSynchronize(TIMER0_BASE, (TIMER_3A_SYNC | TIMER_3B_SYNC | TIMER_4A_SYNC));

    // Enable interrupt for Timer 3A Timeout event
    MAP_IntEnable(INT_TIMER3A);	
    MAP_TimerIntEnable(TIMER3_BASE, TIMER_TIMA_TIMEOUT);

    // Enable interrupt for Timer 4A and B Edge triggered events
    MAP_IntEnable(INT_TIMER2A);
    MAP_IntEnable(INT_TIMER2B);
    MAP_TimerIntEnable(TIMER2_BASE, TIMER_CAPA_EVENT);
    MAP_TimerIntEnable(TIMER2_BASE, TIMER_CAPB_EVENT);

    // Enable the timers
    MAP_TimerEnable(TIMER3_BASE, TIMER_A);
    MAP_TimerEnable(TIMER2_BASE, (TIMER_A | TIMER_B));
}

